/***************************************************************************
 *
 * Copyright (c) 2014 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include "ilm_calibration.h"
#include "ilm_common.h"

#include <stdio.h>
#include <iostream>
#include <vector>
#include <string>
#include <cmath>
#include <cfloat>
#include <cstring>
#include <cstdlib>

char *g_devicename = NULL;

#define DEVICE_NAME g_devicename
#define MAIN_AREA "Main Touch Area"
#define SLIDER_AREA "Volume Slider"

ilmErrorTypes init()
{
    ilmErrorTypes result = ILM_FAILED;

    result = ilm_init();

    if (ILM_SUCCESS == result)
    {
        std::cout << "ilm_init success" << std::endl;
    }
    else
    {
        std::cerr << "ilm_init failed" << std::endl;
    }

    return result;
}

ilmErrorTypes shutdown()
{
    ilmErrorTypes error = ILM_FAILED;
    std::cout << "Calling ilm_destroy... " << std::endl;
    error = ilm_destroy();
    return error;
}

void outputTestResult(std::string& testName, bool testPassed)
{
    if(testPassed)
    {
        std::cout << "PASSED - " << testName << std::endl;
    }
    else
    {
        std::cerr << "FAILED - " << testName << std::endl;
    }
}

void outputParameters(std::string& title, ilmCalibrationParameters& parameters)
{

    std::cout << title << std::endl
              << "x1: " << parameters.x1 << std::endl
              << "x2: " << parameters.x2 << std::endl
              << "x3: " << parameters.x3 << std::endl
              << "y1: " << parameters.y1 << std::endl
              << "y2: " << parameters.y2 << std::endl
              << "y3: " << parameters.y3 << std::endl;

}

bool testCalibrationMatch(const ilmCalibrationParameters& gotParameters,
                          const ilmCalibrationParameters& setParameters)
{
    return ((std::fabs(gotParameters.x1 - setParameters.x1) < DBL_EPSILON)
             && (std::fabs(gotParameters.x2 - setParameters.x2) < DBL_EPSILON)
             && (std::fabs(gotParameters.x3 - setParameters.x3) < DBL_EPSILON)
             && (std::fabs(gotParameters.y1 - setParameters.y1) < DBL_EPSILON)
             && (std::fabs(gotParameters.y2 - setParameters.y2) < DBL_EPSILON)
             && (std::fabs(gotParameters.y3 - setParameters.y3) < DBL_EPSILON));
}

bool testCalibrationParametersAreNotZeroes(const ilmCalibrationParameters& gotParameters)
{
    ilmCalibrationParameters Parameters;

    Parameters.x1 = 0.0;
    Parameters.x2 = 0.0;
    Parameters.x3 = 0.0;

    Parameters.y1 = 0.0;
    Parameters.y2 = 0.0;
    Parameters.y3 = 0.0;

    return !(testCalibrationMatch(gotParameters, Parameters));
}

bool testCoordinateMatch(const ilmCoordinate& firstCoordinate, const ilmCoordinate& secondCoordinate)
{
    return ((firstCoordinate.x == secondCoordinate.x) 
             && (firstCoordinate.y == secondCoordinate.y));
}

bool testCoordinateIsNonZero(const ilmCoordinate& gotCoordinate)
{
    ilmCoordinate zeroCoordinate = {0,0};
 
    return !testCoordinateMatch(gotCoordinate, zeroCoordinate);
}

bool resetCoefficientParametersToZeroes()
{
    ilmCalibrationParameters Parameters;
    ilmErrorTypes error;

    Parameters.x1 = 1.0;
    Parameters.x2 = 0.0;
    Parameters.x3 = 0.0;

    Parameters.y1 = 0.0;
    Parameters.y2 = 1.0;
    Parameters.y3 = 0.0;

    error = ilm_calibrationApplyParameters(DEVICE_NAME, &Parameters);

    return error == ILM_SUCCESS ? true : false;
}

bool resetCoefficientParametersToDefault()
{
    ilmCalibrationParameters Parameters;
    ilmErrorTypes error;

    Parameters.x1 = 1.0;
    Parameters.x2 = 0.0;
    Parameters.x3 = 0.0;

    Parameters.y1 = 0.0;
    Parameters.y2 = 1.0;
    Parameters.y3 = 0.0;

    error = ilm_calibrationApplyParameters(DEVICE_NAME, &Parameters);

    return error == ILM_SUCCESS ? true : false;
}

bool testApplyFromRawCoordinates()
{
    ilmCoordinate expectedLogical1;
    ilmCoordinate expectedLogical2;
    ilmCoordinate expectedLogical3;
    ilmCoordinate sampledRaw1;
    ilmCoordinate sampledRaw2;
    ilmCoordinate sampledRaw3;
    ilmErrorTypes error;
    std::string testName("Apply Calibration from Raw Coordinates");
    bool testResult = false;
    ilmCalibrationParameters calculatedParameters;

    //Texas Instruments Paper case
    //Please see http://www.ti.com/lit/an/slyt277/slyt277.pdf
    //Page 7 - Section Titled: Example 1: Three-point calibration
    expectedLogical1.x = 64;
    expectedLogical1.y = 384;
    expectedLogical2.x = 192;
    expectedLogical2.y = 192;
    expectedLogical3.x = 192;
    expectedLogical3.y = 576;
    sampledRaw1.x = 678;
    sampledRaw1.y = 2169;
    sampledRaw2.x = 2807;
    sampledRaw2.y = 1327;
    sampledRaw3.x = 2629;
    sampledRaw3.y = 3367;

    resetCoefficientParametersToZeroes();

    error = ilm_calibrationApplyFromRawCoordinates(DEVICE_NAME,
                                                   "Main Touch Area",
                                                   &expectedLogical1,
                                                   &expectedLogical2,
                                                   &expectedLogical3,
                                                   &sampledRaw1,
                                                   &sampledRaw2,
                                                   &sampledRaw3);

    if (error == ILM_SUCCESS)
    {
        error = ilm_calibrationGetParameters(DEVICE_NAME,
                                             &calculatedParameters);
        if (error == ILM_SUCCESS)
        {
            if (testCalibrationParametersAreNotZeroes(calculatedParameters))
            {
                testResult = true; 
            }
            else
            {
                testResult = false;
            }
        }
        else
        {
            testResult = false;
        }
    }
    else
    {
        testResult = false;
    }

    outputTestResult(testName, testResult);

    return testResult;
}

bool testApplyFromRawCoordinatesUnitTest1()
{
    ilmCoordinate expectedLogical1;
    ilmCoordinate expectedLogical2;
    ilmCoordinate expectedLogical3;
    ilmCoordinate sampledRaw1;
    ilmCoordinate sampledRaw2;
    ilmCoordinate sampledRaw3;
    ilmErrorTypes error;
    std::string testName("Apply Calibration from Raw Coordinates - Unity Test 1 Calibration");
    bool testResult = false;
    ilmCalibrationParameters calculatedParameters;

    // Unity Calibration
    expectedLogical1.x = 0;
    expectedLogical1.y = 0;
    expectedLogical2.x = 1023;
    expectedLogical2.y = 767;
    expectedLogical3.x = 1023;
    expectedLogical3.y = 0;
    sampledRaw1.x = 500;
    sampledRaw1.y = 2900;
    sampledRaw2.x = 29999;
    sampledRaw2.y = 30999;
    sampledRaw3.x = 29999;
    sampledRaw3.y = 2900;

    resetCoefficientParametersToZeroes();

    error = ilm_calibrationApplyFromRawCoordinates(DEVICE_NAME,
                                                   "Main Touch Area",
                                                   &expectedLogical1,
                                                   &expectedLogical2,
                                                   &expectedLogical3,
                                                   &sampledRaw1,
                                                   &sampledRaw2,
                                                   &sampledRaw3);

    if (error == ILM_SUCCESS)
    {
        error = ilm_calibrationGetParameters(DEVICE_NAME,
                                             &calculatedParameters);
        if (error == ILM_SUCCESS)
        {
            outputParameters(testName, calculatedParameters);

            if (testCalibrationParametersAreNotZeroes(calculatedParameters))
            {
                testResult = true; 
            }
        }
    }

    outputTestResult(testName, testResult);

    return testResult;
}

bool testApplyFromLogicalCoordinates()
{
    ilmCoordinate expectedLogical1;
    ilmCoordinate expectedLogical2;
    ilmCoordinate expectedLogical3;
    ilmCoordinate sampledLogical1;
    ilmCoordinate sampledLogical2;
    ilmCoordinate sampledLogical3;
    ilmErrorTypes error;
    std::string testName("Apply Calibration from Logical Coordinates");
    bool testResult = false;
    ilmCalibrationParameters zeroParameters = {0.0, 0.0, 0.0, 0.0, 0.0, 0.0};

    expectedLogical1.x = 264;
    expectedLogical1.y = 384;
    expectedLogical2.x = 292;
    expectedLogical2.y = 292;
    expectedLogical3.x = 292;
    expectedLogical3.y = 576;
    sampledLogical1.x = 267;
    sampledLogical1.y = 390;
    sampledLogical2.x = 298;
    sampledLogical2.y = 283;
    sampledLogical3.x = 296;
    sampledLogical3.y = 580;

    resetCoefficientParametersToZeroes();

    error = ilm_calibrationApplyFromLogicalCoordinates(DEVICE_NAME,
                                                       "Main Touch Area",
                                                       &expectedLogical1,
                                                       &expectedLogical2,
                                                       &expectedLogical3,
                                                       &sampledLogical1,
                                                       &sampledLogical2,
                                                       &sampledLogical3);

    if (error == ILM_SUCCESS)
    {
        error = ilm_calibrationGetParameters(DEVICE_NAME,
                                             &zeroParameters);
        if (error == ILM_SUCCESS)
        {
            if (testCalibrationParametersAreNotZeroes(zeroParameters))
            {
                testResult = true; 
            }
            else
            {
                testResult = false;
            }
        }
        else
        {
            testResult = false;
        }
    }
    else
    {
        testResult = false;
    }

    outputTestResult(testName, testResult);

    return testResult;
}

bool testApplyFromLogicalCoordinatesUnitTest1()
{
    ilmCoordinate expectedLogical1;
    ilmCoordinate expectedLogical2;
    ilmCoordinate expectedLogical3;
    ilmCoordinate sampledLogical1;
    ilmCoordinate sampledLogical2;
    ilmCoordinate sampledLogical3;
    ilmErrorTypes error;
    std::string testName("Apply Calibration from Logical Coordinates - Unit Test1");
    bool testResult = false;
    ilmCalibrationParameters zeroParameters = {0.0, 0.0, 0.0, 0.0, 0.0, 0.0};

     expectedLogical1.x = 264;
    expectedLogical1.y = 384;
    expectedLogical2.x = 292;
    expectedLogical2.y = 292;
    expectedLogical3.x = 292;
    expectedLogical3.y = 576;
    sampledLogical1.x = 267;
    sampledLogical1.y = 390;
    sampledLogical2.x = 298;
    sampledLogical2.y = 283;
    sampledLogical3.x = 296;
    sampledLogical3.y = 580;

    resetCoefficientParametersToZeroes();

    error = ilm_calibrationApplyFromLogicalCoordinates(DEVICE_NAME,
                                                       "Main Touch Area",
                                                       &expectedLogical1,
                                                       &expectedLogical2,
                                                       &expectedLogical3,
                                                       &sampledLogical1,
                                                       &sampledLogical2,
                                                       &sampledLogical3);

    if (error == ILM_SUCCESS)
    {
        error = ilm_calibrationGetParameters(DEVICE_NAME,
                                             &zeroParameters);
        if (error == ILM_SUCCESS)
        {
            outputParameters(testName, zeroParameters);

            if (testCalibrationParametersAreNotZeroes(zeroParameters)) testResult = true; 
        }
    }

    outputTestResult(testName, testResult);

    return testResult;
}

bool testCalibrationStore()
{
    ilmErrorTypes error;
    std::string testName("Test Calibration Store");
    bool testResult = false;

    error = ilm_calibrationStore(DEVICE_NAME);

    testResult = error == ILM_SUCCESS ? true : false;

    outputTestResult(testName, testResult);

    return testResult;
}

bool testCalibrationRestore()
{
    ilmErrorTypes error;
    std::string testName("Test Calibration Restore");
    bool testResult = false;

    error = ilm_calibrationRestore(DEVICE_NAME);

    testResult = error == ILM_SUCCESS ? true : false;

    return testResult;
}

bool testCalibrationGetParameters(const ilmCalibrationParameters& setParameters)
{
    ilmErrorTypes error;
    ilmCalibrationParameters parameters;
    std::string testName("Test Get Calibration Parameters");
    bool match = false;

    error = ilm_calibrationGetParameters(DEVICE_NAME,
                                         &parameters);

    if (error == ILM_SUCCESS)
    {
        match = testCalibrationMatch(parameters, setParameters);
    }

    outputTestResult(testName, match);

    return match;
}

bool testCalibrationApplyParameters(ilmCalibrationParameters& setParameters)
{
    ilmErrorTypes error;
    std::string testName("Test Apply Calibration Parameters");
    bool testResult = false;

    error = ilm_calibrationApplyParameters(DEVICE_NAME, &setParameters);

    testResult = (error == ILM_SUCCESS) ? true : false;

    outputTestResult(testName, testResult);

    return testResult;
}

bool testCalibrationStoredValidity()
{
    ilmErrorTypes error;
    t_ilm_bool valid = false;
    std::string testName("Test Stored Calibration Validity");
    bool testResult = false;

    error = ilm_calibrationStoredCalibrationIsValid(DEVICE_NAME, &valid);

    if (error == ILM_SUCCESS)
    {
        if (valid)
        {
            std::cout << "Valid Calibration Stored" << std::endl;
        }
        else
        {
            std::cout << "No Valid Calibration Stored" << std::endl;
        }

        testResult = true;
    }
    else
    {
        testResult = false;
    }

    outputTestResult(testName, testResult);

    return testResult;
}

bool testCalibrationGetRawCoordinate()
{
    ilmErrorTypes error;
    ilmCoordinate logical;
    std::string testName("Test Get Raw Coordinate from Logical Coordinate");
    bool testResult = false;
    ilmCoordinate raw = {0,0};

    logical.x = 1;
    logical.y = 1;

    error = ilm_calibrationGetRawCoordinate(DEVICE_NAME,
                                            "Main Touch Area",
                                            &logical,
                                            &raw);

    if (error == ILM_SUCCESS)
    {
        testResult = testCoordinateIsNonZero(raw);
    }
    else
    {
        testResult = false;
    }

    outputTestResult(testName, testResult);

    return testResult;
}

bool testCalibrationGetLogicalCoordinate()
{
    ilmErrorTypes error;
    ilmCoordinate logical = {0, 0};
    ilmCoordinate raw;
    t_ilm_string subdivisionName;
    std::string testName("Test Get Logical Coordinate from Raw Coordinate");
    bool result = true;

    const std::string mainAreaSubdivision(MAIN_AREA);
    const std::string sliderSubdivision(SLIDER_AREA);

    raw.x = 3000;
    raw.y = 2901;

    error = ilm_calibrationGetLogicalCoordinate(DEVICE_NAME,
                                                &raw,
                                                &subdivisionName,
                                                &logical);

    if (error == ILM_SUCCESS)
    {
        if (strcmp(subdivisionName, mainAreaSubdivision.c_str()) != 0)
        {
            std::cerr << "Expecting: " << mainAreaSubdivision
                      << ", Got: " << subdivisionName << std::endl;

            result = false;
        }

        if (!testCoordinateIsNonZero(logical))
        {
            std::cerr << "Expecting non zero coordinates, Got: ("
                      << raw.y << ", " << raw.x << ")" << std::endl;

            result = false;
        }
    }
    else
    {
        result = false;
    }

    raw.x = 500;
    raw.y = 12901;

    logical.x = 0;
    logical.y = 0;

    error = ilm_calibrationGetLogicalCoordinate(DEVICE_NAME,
                                                &raw,
                                                &subdivisionName,
                                                &logical);

    if (error == ILM_SUCCESS)
    {
        if (strcmp(subdivisionName, sliderSubdivision.c_str()) != 0)
        {
            std::cerr << "Expecting: " << sliderSubdivision
                      << ", Got: " << subdivisionName << std::endl;

            result = false;
        }

        if (!testCoordinateIsNonZero(logical))
        {
            std::cerr << "Expecting non zero coordinates, Got: ("
                      << raw.y << ", " << raw.x << ")" << std::endl;

            result = false;
        }
    }
    else
    {
        result = false;
    }

    raw.x = 1;
    raw.y = 1;

    error = ilm_calibrationGetLogicalCoordinate(DEVICE_NAME,
                                                &raw,
                                                &subdivisionName,
                                                &logical);

    if (error == ILM_SUCCESS)
    {
        if (!strlen(subdivisionName) <= 0)
        {
            std::cerr << "Expected no subdivision for raw coordinate but got: "
                      << subdivisionName << std::endl;

            result = false;
        }
    }
    else
    {
        result = false;
    }

    outputTestResult(testName, result);

    return result;
}

bool testAppliedCoefficients()
{
    ilmCalibrationParameters Parameters = {0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    std::string testName("Test Applied Coefficients");
    bool result = false;

    if (ILM_SUCCESS == ilm_calibrationGetParameters(DEVICE_NAME,
                                         &Parameters))
    {
        if (testCalibrationParametersAreNotZeroes(Parameters))
            result = true;
    }

    outputTestResult(testName, result);

    return result;
}

bool testAllInterfaces()
{
    ilmErrorTypes error;
    bool result = true;
    ilmCalibrationParameters testParameters = {1.0, 0.0, 0.0, 0.0, 1.0, 0.0};
    std::string testName("Test All Interfaces");

    // Check application of test parameters
    if (!testCalibrationApplyParameters(testParameters))
        result = false;

    // Check getting of test parameters
    if (!testCalibrationGetParameters(testParameters))
        result = false;

    // Reset the applied coefficients to something invalid
    if (!resetCoefficientParametersToZeroes())
        result = false;

    // Test that applied coefficients can be generated from Raw coordinates
    if (!testApplyFromRawCoordinates())
        result = false;

    // Test that the applied coefficients make sense
    if (!testAppliedCoefficients())
        result = false;

    // Reset the applied coefficients to something invalid
    if (!resetCoefficientParametersToZeroes())
        result = false;

    // Test that applied coefficients can be generated from Logical coordinates
    if (!testApplyFromLogicalCoordinates())
        result = false;

    // Test that the applied coefficients make sense
    if (!testAppliedCoefficients())
        result = false;

    // Store the coefficients to disc
    if (!testCalibrationStore())
        result = false;

    // Reset the applied coefficients to something invalid
    if (!resetCoefficientParametersToZeroes())
        result = false;

    // Restore the valid calibrations from disc
    if (!testCalibrationRestore())
         result = false;

    // Test that the Restored coefficients make sense
    if (!testAppliedCoefficients())
         result = false;

    // Reset Coefficients to something unity
    resetCoefficientParametersToDefault();

    // Test that sensible raw coordinates can be got
    if (!testCalibrationGetRawCoordinate())
         result = false;

    // Test that sensible logical coordinates can be got
    if (!testCalibrationGetLogicalCoordinate())
         result = false;

    outputTestResult(testName, result);

    return result;
}

int main(int argc, char **argv)
{
    ilmErrorTypes error;
    bool result = true;
    bool interfaceTest = true;
    std::string type;

    if (argc > 2)
    {
        type = argv[1];

        if (type == "UNIT") interfaceTest = false;
        else if (type == "INTERFACE") interfaceTest = true;
        else
        {
            std::cerr << "Usage: '" << argv[0] << " UNIT' 'DEVICE NAME' or '"
                      << argv[0] << " INTERFACE' 'DEVICE NAME' " << std::endl;
            return EXIT_FAILURE;
        }
        g_devicename = argv[2];
    }
    else
    {
	std::cerr << "Usage: '" << argv[0] << " UNIT' 'DEVICE NAME' or '"
			  << argv[0] << " INTERFACE' 'DEVICE NAME' " << std::endl;
	return EXIT_FAILURE;

    }

    if (ILM_SUCCESS != init())
    {
        return EXIT_FAILURE;
    }


    if (interfaceTest)
    {
        if (!testApplyFromRawCoordinates())
            result = false;

        if (!testApplyFromLogicalCoordinates())
            result = false;

        if (!testCalibrationStore())
            result = false;

        if (!testCalibrationRestore())
            result = false;

        ilmCalibrationParameters setParameters;

        setParameters.x1 = 1.0;
        setParameters.x2 = 2.0;
        setParameters.x3 = 3.0;
        setParameters.y1 = 4.0;
        setParameters.y2 = 5.0;
        setParameters.y3 = 6.0;

        if (!testCalibrationApplyParameters(setParameters))
            result = false;

        if (!testCalibrationGetParameters(setParameters))
            result = false;

        if (!testCalibrationStoredValidity())
            result = false;

        resetCoefficientParametersToDefault();

        if (!testCalibrationGetRawCoordinate())
            result = false;

        if (!testCalibrationGetLogicalCoordinate())
            result = false;

        // Test Overall behaviour makes sense
        if (!testAllInterfaces())
            result = false;
    }
    else
    {
        if (!testApplyFromRawCoordinatesUnitTest1())
            result = false;

        if (!testApplyFromLogicalCoordinatesUnitTest1())
            result = false;
    }

    if (ILM_SUCCESS != shutdown())
    {
        return EXIT_FAILURE;
    }

    return 0;
}
